/*
 * Routines for handling event notification.
 */
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

#include "libfma.h"
#include "lf_fms_comm.h"

#include "fms.h"
#include "fms_error.h"
#include "fms_notify.h"

/*
 * local routines
 */
static int get_notify_id(void);

/*
 * Initialize notification subsystem and establish stderr
 */
int
init_notify_vars()
{
  LF_CALLOC(F.notify, struct fms_notify, 1);
  return 0;

 except:
  return -1;
}

/*
 * Get the next ID for a notification client.  Do the realloc carefully such
 * that failure does not leave us in an inconsistant state.
 */
static int
get_notify_id()
{
  int id;
  void *p;
  int n;

  for (id=0; id<F.notify->num_notify_slots; ++id) {
    if (F.notify->clients[id] == NULL) {
      LF_CALLOC(F.notify->clients[id], struct fms_notify_client, 1);
      return id;
    }
  }

  /* if we get here, no room for this client.  allocate more room */
  n = F.notify->num_notify_slots + FMS_NOTIFY_CLIENT_ARRAY_INCR;
  p = realloc(F.notify->clients, sizeof(struct fms_notify_client *) * n);
  if (p == NULL) LF_ERROR(("allocating additional notify client space"));

  /* copy the new pointer and return the new id */
  F.notify->num_notify_slots = n;
  F.notify->clients = p;

  /* clear the new section of the array */
  memset(&F.notify->clients[id],
         0,
         sizeof(struct fms_notify_client *) *FMS_NOTIFY_CLIENT_ARRAY_INCR);

  /* id is still good - fill it in and return */
  LF_CALLOC(F.notify->clients[id], struct fms_notify_client, 1);
  return id;

 except:
  return -1;
}

/*
 * Register to send events to a file
 */
int
register_notify_file(
  int events,
  FILE *fp)
{
  int id;
  struct fms_notify_client *ncp;

  id = get_notify_id();
  if (id == -1) LF_ERROR(("Error registering file notification"));

  /* fill in the notify struct */
  ncp = F.notify->clients[id];
  ncp->notify_type = FMS_NOTIFY_FILE;
  ncp->event_types = events;
  ncp->u.file = fp;
  return 0;

 except:
  return -1;
}

/*
 * Register to send events via exec
 */
int
register_notify_exec(
  int events,
  char *program)
{
  int id;
  struct fms_notify_client *ncp;

  id = get_notify_id();
  if (id == -1) LF_ERROR(("Error registering exec notification"));

  /* fill in the notify struct */
  ncp = F.notify->clients[id];
  ncp->notify_type = FMS_NOTIFY_EXEC;
  ncp->event_types = events;
  LF_DUP_STRING(ncp->u.program, program);
  return 0;

 except:
  return -1;
}

/*
 * Register to send events via email
 */
int
register_notify_email(
  int events,
  char *email)
{
  int id;
  struct fms_notify_client *ncp;

  id = get_notify_id();
  if (id == -1) LF_ERROR(("Error registering email notification"));

  /* fill in the notify struct */
  ncp = F.notify->clients[id];
  ncp->notify_type = FMS_NOTIFY_EMAIL;
  ncp->event_types = events;
  LF_DUP_STRING(ncp->u.email, email);
  return 0;

 except:
  return -1;
}

/*
 * return a string for a severity
 */
static inline char *
event_severity_string(
  int type)
{
  return (type == FMS_EVENT_INFO) ? "INFO" :
         (type == FMS_EVENT_ALERT) ? "ALERT" :
         (type == FMS_EVENT_ERROR) ? "ERROR" :
         (type == FMS_EVENT_CRITICAL) ? "CRITICAL ERROR" :
	 "DEBUG";
}

/*
 * Notify all the proper parties of an event. Takes a variable
 * number of args.
 */
void
fms_notify(
  int event_type,
  const char *fmt, ...)
{
  va_list args;
  int n;
  char *p = NULL;
  int size = LF_STRING_LEN;
  int tries = 0;
  int id;
  struct fms_notify_client *ncp;

  LF_CALLOC(p, char, size);

  while (tries++ < 2) {
    va_start (args, fmt);
    n = vsnprintf (p, size, fmt, args);
    va_end (args);

    if (n > -1 && n < size) {
      break;
    } else {
      /* try again with more space */
      size *= 2;
      p = realloc(p, size);   /* times 2 for power of 2 style mallocs */
      if (p  == NULL) {
	LF_ERROR(("Error reallocing print buffer"));
      }
    }
  }

  /*
   * loop through all registrations, notifying those that match
   */
  for (id=0; id<F.notify->num_notify_slots; ++id) {

    ncp = F.notify->clients[id];
    if (ncp == NULL) continue;
    if ((ncp->event_types & event_type) == 0) continue;

    /* got a match - perform the notification */
    switch (ncp->notify_type) {

    case FMS_NOTIFY_FILE:
      fprintf(ncp->u.file, "%s %s: %s\n",
	  lf_timestamp(), event_severity_string(event_type), p);
      fflush(ncp->u.file);
      break;

    case FMS_NOTIFY_EMAIL:
      {
	FILE *pp;
	char cmd[256];

        printf("sending mail to %s\n", ncp->u.email);
	sprintf(cmd, "/usr/lib/sendmail %s", ncp->u.email);
	pp = popen(cmd, "w");
	if (pp == NULL) LF_ERROR(("execing sendmail"));
	
	fprintf(pp, "From: FMS\n\n");
	fprintf(pp, "%s: %s\n", event_severity_string(event_type), p);
	fclose(pp);
      }
      break;

    case FMS_NOTIFY_EXEC:
      {
	FILE *pp;
	lf_string_t cmd;

	sprintf(cmd, "%s", ncp->u.program);
	pp = popen(cmd, "w");
	if (pp == NULL) LF_ERROR(("Error execing %s", ncp->u.program));
	
	fprintf(pp, "%s\n%s\n", event_severity_string(event_type), p);
	fclose(pp);
      }
      break;
    }
  }
  LF_FREE(p);
  return;

 except:
  fms_perror();
  fms_exit(1);
  return;
}
